 // 8/10/2018


//////////////////////////////////////////////////////////////////////
//
//  Some auxiliary functions
//
//////////////////////////////////////////////////////////////////////

// Split Cartan -- diagonal matrices in GL(2,Z/nZ) -- E admits two distinct rational n-isogenies if and only if \im\rho_{E,n} is conjugate ot a subgroup of the split Cartan
function GL2SplitCartan(R)
    if Type(R) eq RngIntElt then R:=Integers(R); end if;
    M,pi:=MultiplicativeGroup(R);
    return sub<GL(2,R) | [[pi(g),0,0,1] : g in Generators(M)], [[1,0,0,pi(g)] : g in Generators(M)]>;
end function;

// Borel group -- upper triangular matrices in GL(2,Z/nZ) -- E admits a rational n-isogeny if and only if \im\rho_{E,n} is conjugate to a subgroup of  the Borel
function GL2Borel(R)
    if Type(R) eq RngIntElt then R:=Integers(R); end if;
    return sub<GL(2,R) | [1,1,0,1],GL2SplitCartan(R)>;
end function;

// Subgroup of Borel that fixes a point of order n
function GL2Borel1(R)
    if Type(R) eq RngIntElt then R:=Integers(R); end if;
    M,pi:=MultiplicativeGroup(R);    
    return sub<GL(2,R) | [1,1,0,1], [[1,0,0,pi(g)] : g in Generators(M)]>;
end function;

// Checks whether H1 is conjugate to a subgroup of H2 in G
function IsConjugateToSubgroup(G,H1,H2)
    if not IsDivisibleBy(#H2,#H1) then return false; end if;
    n:=#H2 div #H1;
    return #[H:H in Subgroups(H2:IndexEqual:=n)|IsConjugate(G,H`subgroup,H1)] ne 0;
end function;

//////////////////////////////////////////////////////////////////////
//
//  .......
//
//////////////////////////////////////////////////////////////////////


function LiftGroups(G1,G2)
    N1 := #BaseRing(G1);
    N2 := #BaseRing(G2);
    N := N1*N2;
    GLN := GL(2,Integers(N));
    GLN1,mp1 := ChangeRing(GLN,Integers(N1));
    GLN2,mp2 := ChangeRing(GLN,Integers(N2));
    gens1 := [Inverse(mp1)(g): g in Generators(G1)];
    gens2 := [Inverse(mp2)(g): g in Generators(G2)];
    Append(~gens1,-Identity(GLN));
    Append(~gens2,-Identity(GLN));
    for g in Generators(Kernel(mp1)) do
        Append(~gens1,g);
    end for;
    for g in Generators(Kernel(mp2)) do
        Append(~gens2,g);
    end for;
    GL1 := sub<GLN | gens1>;
    GL2 := sub<GLN | gens2>;
    return GL1 meet GL2;
end function;

//////////////////////////////////////////////////////////////////////
//
//  Functions to discard cases using 
//  Q-isogenies degrees and torsion type (Mazur's Theorems)
//
//////////////////////////////////////////////////////////////////////


NonCMIsogDegrees := [ 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 15, 16, 17, 18, 21, 25, 37];

PossibleTorsion := [ 
[ 1, 1 ], 
[ 1, 2 ],
[ 1, 3 ], 
[ 1, 4 ], 
[ 1, 5 ], 
[ 1, 6 ], 
[ 1, 7 ], 
[ 1, 8 ], 
[ 1, 9 ], 
[ 1, 10 ], 
[ 1, 12 ],
[ 2, 2 ],
[ 2, 4 ],
[ 2, 6 ],
[ 2, 8 ] ];

function IsogenyDegree(G)
	R := BaseRing(G);
    if IsConjugateToSubgroup( GL(2,R), G, GL2SplitCartan(R)) then return #R^2; end if;
    if IsConjugateToSubgroup( GL(2,R), G, GL2Borel(R)) then return #R; end if;
    return 1;
end function;

function TorsionType(G)
    R := BaseRing(G); p := #R;
    //assert IsPrime(p);
    if #G eq 1 then return [p,p]; end if;
    if IsConjugateToSubgroup(GL(2,R), G, GL2Borel1(R)) then return [1,p]; end if;
    return [1,1];
end function;



//////////////////////////////////////////////////////////////////////
//
// The following functions come from SZ16.
//
//////////////////////////////////////////////////////////////////////


// Given a subgroup G of GL(2,Z/nZ), return nu2(Gamma), where Gamma is the PSL(2,Z) congruence subgroup Gamma corresponding to G (which depends only on the intersection of G with SL(2,/Z/nZ) mod +/-1)
// That is, we take the inverse image of G meet SL(2,Z/nZ) in SL(2,Z)/{-1} and count the number of elliptic points of order 2, i.e. Gamma-inequivalent SL(2,Z)-translates of i that have non-trivial stabilizers in Gamma
// This amounts to counting right cosets fixed by the stablizer of i, which is generated by e2=[0,1,-1,0]
function GL2Nu2(G)
    SL2 := SL(2,BaseRing(G));
    H := sub<SL2|G meet SL2,[-1,0,0,-1]>;
    R:=RightTransversal(SL2,H);
    e2:=SL2![0,1,-1,0];
    return #{g:g in R|g*e2*g^-1 in H};
end function;
    
// Given a subgroup G of GL(2,Z/nZ), return nu3(Gamma), where Gamma is the PSL(2,Z) congruence subgroup Gamma corresponding to G (which depends only on the intersection of G with SL(2,/Z/nZ) mod +/-1)
// That is, we take the inverse image of G meet SL(2,Z/nZ) in SL(2,Z)/{-1} and count the number of elliptic points of order 3, i.e. Gamma-inequivalent SL(2,Z)-translates of rho that have non-trivial stabilizers in Gamma
// This amounts to counting right cosets fixed by the stablizer of z_3, which is generated by e2=[0,1,-1,0]
function GL2Nu3(G)
    SL2 := SL(2,BaseRing(G));
    H := sub<SL2|G meet SL2,[-1,0,0,-1]>;
    R:=RightTransversal(SL2,H);
    e3:=SL2![0,1,-1,-1];
    return #{g:g in R|g*e3*g^-1 in H};
end function;

// Given a subgroup G of GL(2,Z/nZ), return the number of cusp orbits for Gamma, where Gamma is the PSL(2,Z) congruence subgroup Gamma corresponding to G (which depends only on the intersection of G with SL(2,/Z/nZ) mod +/-1)
// That is, we take the inverse image of G meet SL(2,Z/nZ) in SL(2,Z)/{-1} and count the number of orbits of infty, equivalently, the number of orbits of the right-coset space of Gamma under the right-action of T=[1,1,0,1]
function GL2CuspCount(G)
    SL2 := SL(2,BaseRing(G));
    H := sub<SL2|G meet SL2,[-1,0,0,-1]>;
    R:=[r:r in RightTransversal(SL2,H)];
    T:=SL2![1,1,0,1];
    c:=0;
    while #R ne 0 do
        // pick a right coset rep
        r := R[#R];  R:=Prune(R); c+:=1;
        // compute its orbit under right-multiplication by T, removing each coset rep in the orbit from R as we go
        while true do
            r := r*T;
            S:=[i:i in [1..#R]|R[i]*r^-1 in H]; // figure out which coset in our list is equal to the coset represented by r
            if #S eq 0 then break; end if;     // if we can't find it, its because we already removed it (i.e we cycled back to our starting point)
            assert #S eq 1;
            R:=Remove(R,S[1]);
        end while;
    end while;
    return c;
end function;

// Given a subgroup G of GL(2,Z/nZ), return the genus of the modular curve X_G', where G'=<G,-I>
function GL2Genus(G)
    SL2:=SL(2,BaseRing(G));
    H:=sub<SL2|G meet SL2,[-1,0,0,-1]>;
    g := 1 + Index(SL2,H)/12 - GL2Nu2(H)/4 - GL2Nu3(H)/3 - GL2CuspCount(H)/2;
    return Integers()!g;
end function;

//////////////////////////////////////////////////////////////////////
//
//  j-maps from Zywina + Dokchister^2 + Elkies
//
//////////////////////////////////////////////////////////////////////


F<t> := FunctionField(Rationals());
grp := recformat< level:IntegerRing(), group:GrpMat, jmap:FldFunRatUElt>;
Groups := AssociativeArray();
Groups["2Cs"] := rec<grp| level:=2, group:=sub<GL(2,Integers(2)) | {[1,0,0,1]}>, jmap := 256*(t^2+t+1)^3/(t^2*(t+1)^2)>; 
Groups["2B"] := rec<grp| level:=2, group:=sub<GL(2,Integers(2)) | {[1,1,0,1]}>, jmap := 256*(t+1)^3/t>;
Groups["2Cn"] := rec<grp| level:=2, group:=sub<GL(2,Integers(2)) | {[1,1,1,0]}>, jmap := t^2+1728>;

X3:=sub<GL(2,Integers(4)) | {[3,3,0,1],[0,1,3,1]}>;
X4:=sub<GL(2,Integers(8)) | {[7,7,0,1],[5,0,1,1]}>;
X5:=sub<GL(2,Integers(8)) | {[5,5,0,1],[0,1,1,1]}>;
X7:=sub<GL(2,Integers(4)) | {[0,1,3,0],[0,1,1,1]}>; 

Groups["4X7"] := rec<grp| level:=4,group:=X7, jmap :=-4*t^3*(t+8)>;
Groups["4X3"] := rec<grp| level:=4,group:=X3,jmap :=-t^2+1728>;
Groups["8X4"] := rec<grp| level:=8,group:=X4,jmap :=-2*t^2+1728>;
Groups["8X5"] := rec<grp| level:=8,group:=X5,jmap :=2*t^2+1728>;



Groups["3Cs"] := rec<grp| level:=3, group:=sub<GL(2,Integers(3)) | {[-1,0,0,-1],[1,0,0,2]}>, jmap := 27*(t+1)^3*(t+3)^3*(t^2+3)^3/(t^3*(t^2+3*t+3)^3)>;
Groups["3Ns"] := rec<grp| level:=3, group:=sub<GL(2,Integers(3)) | {[1,0,0,2],[2,0,0,1],[0,-1,1,0]}>, jmap := 27*(t+1)^3*(t-3)^3/t^3>;
Groups["3B"] := rec<grp| level:=3, group:=sub<GL(2,Integers(3)) | {[1,1,0,1],[2,0,0,1],[1,0,0,2]}>, jmap := 27*(t+1)*(t+9)^3/t^3>;
Groups["3Nn"] := rec<grp| level:=3, group:=sub<GL(2,Integers(3)) | {[1,0,0,-1],[-1,0,0,-1],[0,-1,1,0],[1,1,-1,1]}>, jmap := t^3>;

XE:=sub<GL(2,Integers(9))| {[4,5,4,4], [4,8,8,6]}>;

Groups["9XE"] := rec<grp| level:=9, group:=XE, jmap:=(3^7*(t^2-1)^3*(t^6+3*t^5+6*t^4+t^3-3*t^2+12*t+16)^3*(2*t^3+3*t^2-3*t-5))/(t^3-3*t-1)^9>;


Groups["5Cs.4.1"] := rec<grp| level:=5, group:=sub<GL(2,Integers(5)) | {[-1,0,0,-1], [1,0,0,2]}>, jmap := (t^20+228*t^15+494*t^10-228*t^5+1)^3/(t^5*(t^10-11*t^5-1)^5)>;
Groups["5Cs"] := rec<grp| level:=5, group:=sub<GL(2,Integers(5)) | {[2,0,0,1], [1,0,0,2]}>, jmap := (t^2+5*t+5)^3*(t^4+5*t^2+25)^3*(t^4+5*t^3+20*t^2+25*t+25)^3/(t^5*(t^4+5*t^3+15*t^2+25*t+25)^5)>;
Groups["5Ns.2.1"] := rec<grp| level:=5, group:=sub<GL(2,Integers(5)) | {[0,6,3,0],[2,0,0,2], [1,0,0,-1]}>, jmap := 5^4*t^3*(t^2+5*t+10)^3*(2*t^2+5*t+5)^3*(4*t^4+30*t^3+95*t^2+150*t+100)^3/((t^2+5*t+5)^5*(t^4+5*t^3+15*t^2+25*t+25)^5)>;
Groups["5Ns"] := rec<grp| level:=5, group:=sub<GL(2,Integers(5)) | {[2,0,0,1], [1,0,0,2],[0,-1,1,0]}>, jmap := (t+5)^3*(t^2-5)^3*(t^2+5*t+10)^3/(t^2+5*t+5)^5>;
Groups["5B.4.2"] := rec<grp| level:=5, group:=sub<GL(2,Integers(5)) | {[-1,0,0,-1], [1,1,0,1], [2,0,0,1]}>, jmap := ((t^4+228*t^3+494*t^2-228*t+1)^3/(t*(t^2-11*t-1)^5))>;
Groups["5B.4.1"] := rec<grp| level:=5, group:=sub<GL(2,Integers(5)) | {[-1,0,0,-1], [1,1,0,1], [1,0,0,2]}>, jmap := (t^4-12*t^3+14*t^2+12*t+1)^3/(t^5*(t^2-11*t-1))>;
Groups["5Nn"] := rec<grp| level:=5, group:=sub<GL(2,Integers(5)) | {[1,2,1,1],[1,6,3,1], [1,0,0,-1]}>, jmap := 5^3*(t+1)*(2*t+1)^3*(2*t^2-3*t+3)^3/(t^2+t-1)^5>;
Groups["5B"] := rec<grp| level:=5, group:=sub<GL(2,Integers(5)) | {[2,0,0,1],[1,0,0,2],[1,1,0,1]}>, jmap := 5^2*(t^2+10*t+5)^3/t^5>;
Groups["5S4"] := rec<grp| level:=5, group:=sub<GL(2,Integers(5)) | {[2,0,0,1], [1,0,0,2],[0,-1,1,0],[1,1,1,-1]}>, jmap :=  t^3*(t^2+5*t+40) >;

//Groups["7Ns.3.1"] := rec<grp| level:=7, group:=sub<GL(2,Integers(7)) | {[2,0,0,4], [0,2,1,0],[-1,0,0,-1]}>, jmap := F!3^3*5*7^5/2^7>;
Groups["7Ns"] := rec<grp| level:=7, group:=sub<GL(2,Integers(7)) | {[3,0,0,1],[1,0,0,3],[0,-1,1,0]}>, jmap := t*(t+1)^3*(t^2-5*t+1)^3*(t^2-5*t+8)^3*(t^4-5*t^3+8*t^2-7*t+7)^3/(t^3-4*t^2+3*t+1)^7>;
Groups["7B.6.1"] := rec<grp| level:=7, group:=sub<GL(2,Integers(7)) | {[-1,0,0,-1],[1,1,0,1],[1,0,0,3]}>, jmap := (t^2-t+1)^3*(t^6-11*t^5+30*t^4-15*t^3-10*t^2+5*t+1)^3/((t-1)^7*t^7*(t^3-8*t^2+5*t+1))>;
Groups["7B.6.3"] := rec<grp| level:=7, group:=sub<GL(2,Integers(7)) | {[-1,0,0,-1],[1,1,0,1],[3,0,0,1]}>, jmap := (t^2-t+1)^3*(t^6+229*t^5+270*t^4-1695*t^3+1430*t^2-235*t+1)^3/((t-1)*t*(t^3-8*t^2+5*t+1)^7)>;
Groups["7B.6.2"] := rec<grp| level:=7, group:=sub<GL(2,Integers(7)) | {[1,1,0,1],[3,0,0,3],[1,0,0,-1]}>, jmap := -(t^2-3*t-3)^3*(t^2-t+1)^3*(3*t^2-9*t+5)^3*(5*t^2-t-1)^3/((t^3-2*t^2-t+1)*(t^3-t^2-2*t+1)^7)>;
Groups["7Nn"] := rec<grp| level:=7, group:=sub<GL(2,Integers(7)) | {[1,-4,4,1],[1,0,0,-1]}>, jmap := 64*t^3*(t^2+7)^3*(t^2-7*t+14)^3*(5*t^2-14*t-7)^3/(t^3-7*t^2+7*t+7)^7>;
Groups["7B"] := rec<grp| level:=7, group:=sub<GL(2,Integers(7)) | {[3,0,0,1],[1,0,0,3],[1,1,0,1]}>, jmap := (t^2+245*t+2401)^3*(t^2+13*t+49)/t^7>;

//Groups["11B.10.4"] := rec<grp| level:=11, group:=sub<GL(2,Integers(11)) | {[-1,0,0,-1],[1,1,0,1],[4,0,0,6]}>, jmap := F!-11^2>;
//Groups["11B.10.5"] := rec<grp| level:=11, group:=sub<GL(2,Integers(11)) | {[-1,0,0,-1],[1,1,0,1],[5,0,0,7]}>, jmap := F!-11*131^3>;
A11:=(t^5-9*t^4+17*t^3+20*t^2-73*t+43)^11;
B11:=-(t^2+3*t-6)^3*(108000*t^49 + 23793840*t^48 - 413223722*t^47 - 5377010368*t^46 + 230799738529*t^45 - 3137869050351*t^44 + 23205911712335*t^43 - 90936268647246*t^42 + 33563647471596*t^41 + 1631415220074871*t^40 - 7744726079195413*t^39 - 3218815397602111*t^38 + 236712051437217644*t^37 - 1686428698022253344*t^36 + 7984804002023063554*t^35 - 30444784135263860996*t^34 + 96849826504401032248*t^33 - 232064394883539673213*t^32 + 210175535413395353857*t^31 + 1609695806324946484826*t^30 - 11768533689837648360109*t^29 + 48291196122826259771817*t^28 - 143943931899306373170309*t^27 + 315827025781563232420857*t^26 - 421596979720485992629121*t^25 - 234929885880162547645306*t^24 + 3668241437553022801950917*t^23 - 14221091463553801024770599*t^22 + 39148264563215734730610917*t^21 - 87534472061810348609315974*t^20 + 166474240219619575379485393*t^19 - 275040771573054834247036345*t^18 + 399144725377223909937142938*t^17 - 511840960382358144595839458*t^16 + 581656165535334214665717816*t^15 - 586206578096981243980668654*t^14 + 523465655841901079370457175*t^13 - 413200824632802503354807972*t^12 + 287270832775316643952335709*t^11 - 175049577131269087795781453*t^10 + 92916572268973769104815620*t^9 - 42636417323385892254033027*t^8 + 16754292456737738144357709*t^7 - 5570911068111617263502302*t^6 + 1542648801995330874184236*t^5 - 347819053424928336793068*t^4 + 61683475328903338239178*t^3 - 8117056250720937228985*t^2 + 708318740340941449799*t - 30857360406231018655);
C11:=(4*t-5)*(t^2+3*t-6)^6*(9*t^2-28*t+23)^3*(t^4-5*t^3+74*t^2-245*t+223)^3*(4*t^4-9*t^3-t^2+21*t-32)^3*(25*t^4-114*t^3+167*t^2-86*t+20)^3;
     
Groups["11Nn"] :=rec<grp|level:=11, group:=sub<GL(2,Integers(11)) | {[1,-4,4,1], [1,0,0,-1]}>,jmap:=F!0 >; //This is a genus 1 curve and I need to think a little harder about how to work with it.

P13:=[
	t^12+231*t^11+269*t^10-3160*t^9+6022*t^8-9616*t^7+21880*t^6-34102*t^5+28297*t^4-12455*t^3+2876*t^2-243*t+1,
	t^12-9*t^11+29*t^10-40*t^9+22*t^8-16*t^7+40*t^6-22*t^5-23*t^4+25*t^3-4*t^2-3*t+1,
	(t^4-t^3+2*t^2-9*t+3)*(3*t^4-3*t^3-7*t^2+12*t-4)*(4*t^4-4*t^3-5*t^2+3*t-1),
	t^8+235*t^7+1207*t^6+955*t^5+3840*t^4-955*t^3+1207*t^2-235*t+1,
	t^8-5*t^7+7*t^6-5*t^5+5*t^3+7*t^2+5*t+1,
	t^4+7*t^3+20*t^2+19*t+1
	];
Q13:=[
	t^12-512*t^11-13079*t^10-32300*t^9-104792*t^8-111870*t^7-419368*t^6+111870*t^5-104792*t^4+32300*t^3-13079*t^2+512*t+1,
	t^12-8*t^11+25*t^10-44*t^9+40*t^8+18*t^7-40*t^6-18*t^5+40*t^4+44*t^3+25*t^2+8*t+1
	];

Groups["13B.5.2"] := rec<grp| level:=13, group:=sub<GL(2,Integers(13)) | {[2,0,0,1],[1,1,0,1],[1,0,0,8]}>, jmap := (t^2-t+1)^3*P13[1]^3/((t-1)*t*(t^3-4*t^2+t+1)^13)>;
Groups["13B.5.1"] := rec<grp| level:=13, group:=sub<GL(2,Integers(13)) | {[8,0,0,1],[1,1,0,1],[1,0,0,2]}>, jmap := (t^2-t+1)^3*P13[2]^3/((t-1)^13*t^13*(t^3-4*t^2+t+1))>;
Groups["13B.5.4"] := rec<grp| level:=13, group:=sub<GL(2,Integers(13)) | {[2,0,0,2],[1,1,0,1],[1,0,0,8]}>, jmap := -13^4*(t^2-t+1)^3*P13[3]^3/((t^3-4*t^2+t+1)^13*(5*t^3-7*t^2-8*t+5))>;
Groups["13B.4.2"] := rec<grp| level:=13, group:=sub<GL(2,Integers(13)) | {[2,0,0,1],[1,1,0,1],[1,0,0,4]}>, jmap := (t^4-t^3+5*t^2+t+1)*P13[4]^3/(t*(t^2-3*t-1)^13)>;
Groups["13B.4.1"] := rec<grp| level:=13, group:=sub<GL(2,Integers(13)) | {[4,0,0,1],[1,1,0,1],[1,0,0,2]}>, jmap := (t^4-t^3+5*t^2+t+1)*P13[5]^3/(t^13*(t^2-3*t-1))>;
Groups["13B"] := rec<grp| level:=13, group:=sub<GL(2,Integers(13)) | {[2,0,0,1],[1,1,0,1],[1,0,0,2]}>, jmap := (t^2+5*t+13)*P13[6]^3/t>;

// There are three know j-invariants such that G_E(13)=13S4. Then it is not possible to create Groups["13S4"] in the REC format Groups.
//Groups["13S4"] := rec<grp|level:=13,group:=sub<GL(2,Integers(13)) | {[2,0,0,2], [2,0,0,3], [0,-1,1,0], [1,1,-1,1]}>, jmap:=[ (2^4*5*13^4*17^3)/3^13, -(2^12*5^3*11*13^4)/3^13, (2^18*3^3*13^4*127^3*139^3*157^3*283^3*929)/(5^13*61^13) ]>;

//Groups["17B.4.2"] := rec<grp|level:=17,group:=sub<GL(2,Integers(17)) | {[2,0,0,11], [4,0,0,-4], [1,1,0,1]}>,jmap:=F!-17*373^3/2^17>;
//Groups["17B.4.6"] := rec<grp|level:=17,group:=sub<GL(2,Integers(17)) | {[11,0,0,2], [-4,0,0,4], [1,1,0,1]}>, jmap:=F!-17^2*101^3/2>;

//Groups["37B.8.1"] := rec<grp|level:=37,group:=sub<GL(2,Integers(37)) | {[2^3,0,0,1], [1,0,0,2], [1,1,0,1]}>, jmap:=F!-7*11^3>;
//Groups["37B.8.2"] := rec<grp|level:=37,group:=sub<GL(2,Integers(37)) | {[2,0,0,1], [1,0,0,2^3], [1,1,0,1]}>, jmap:=F!-7*137^3*2083^3>;


//////////////////////////////////////////////////////////////////////
//
//  FP_curve function
//
//////////////////////////////////////////////////////////////////////

function FiberProduct_Curve(g1,g2)
    R<x,y>:=PolynomialRing(Rationals(),2);
	if g1 ne "11Nn" and g2 ne "11Nn" then
		f1 := Groups[g1]`jmap;
       	f2 := Groups[g2]`jmap;
      	g:=Numerator(Evaluate(f1,x)-Evaluate(f2,y));
	  else
	    if g1 eq "11Nn" then
 			f2 := Groups[g2]`jmap;
    		jE:=Evaluate(f2,y);
    		g:=Numerator(Evaluate(A11,x)*jE^2+Evaluate(B11,x)*jE+Evaluate(C11,x));
		end if;
		if g2 eq "11Nn" then
 			f1 := Groups[g1]`jmap;
        	jE:=Evaluate(f1,y);
        	g:=Numerator(Evaluate(A11,x)*jE^2+Evaluate(B11,x)*jE+Evaluate(C11,x));
		end if;				  
    end if;

	return ProjectiveClosure(Curve(AffineSpace(R),g));

end function;

